''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'
' Simple signal generator program for the AD9850 DDS module using
' a 125MHz MCLK source
' written by Jim Rowe
' last updated 30/12/2016 at 7:45 am
' Requires MMBasic 5.1 or later and an ILI9341 based LCD panel with touch
' together with a Micromite BackPack
  
' Note that Micromite's SPI interface uses these pins:
' SPI Function        Micromite pin
' MOSI (SPI out)      Pin 3
' MISO (SPI in)       Pin 14
' SCK (SPI clock)     Pin 25
' SS/CS-bar/LOAD      Pin 22 (used by AD9850 as FQ_UD)
'
' Also note that the SPI interface of the AD9850 DDS chip has the
' following requirements:
' 1. serial data on the SDATA/MOSI line is clocked into the
'    AD9850 input register on the RISING edge of SCK pulses.
' 2. The serial data must be sent to the AD9850 LSB FIRST, i.e., in
'    reversed bit order to the SPI standard. The 32-bit frequency word
'    must also be sent first, before the 8-bit control/phase word.
' 3. the FQ_UD (freq update/SS) input is active HIGH, and this line
'    should be kept low during the 40-SCK pulses used to shift the
'    frequency/phase data word into the AD9850. It should be taken HIGH
'    to load the data into into the AD9850 no sooner than 10ns after
'    the last (40th) rising edge of SCK. It should be kept high for
'    at least 10ns afterwards, before returning it low.
' 4. So for correct operation, the SPI MODE must be set for 0
'    i.e., CPOL = 0 and CPHA = 0.
' 5. Control bits B32 and B33 of the serial programming word should be
'    be kept low (0) at all times, for normal operation.
' 6. Similarly bit B34 should be kept low for normal operation, and only
'    taken high to reset the AD9850's registers.
' 
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  OPTION AUTORUN ON
  Option Explicit
  
  DIM AS FLOAT OutFreq! = 100000  ' initial output frequency in Hertz

  DIM AS INTEGER Btn = 0
  DIM AS INTEGER Ctr = 0          ' counter used for looping, etc
  DIM AS INTEGER IntFlag = 0      ' flag to show if screen has been touched
  DIM AS INTEGER FRegVal = 0      ' calc value to send to DDS freq reg
  DIM AS INTEGER outData1, outData2, inData  ' SPI data variables
  DIM AS INTEGER revData1, revData2          ' for bit reversals
  
DIM AS STRING CurrFreq$           ' current frequency as a string
DIM AS STRING NuFreq$             ' string for nominated new frequency
DIM AS STRING KPInStr$ LENGTH 9   ' input string from touch screen
DIM AS STRING KPStr$              ' input char from InCharFrmKP subroutine
DIM AS STRING str1$ LENGTH 8      ' these strings are for bit reversals
DIM AS STRING str2$ LENGTH 32
  
  CONST FoutFactor! = 0.02910383  ' factor for getting Fout from OutFreq!
                                  ' (equals FmClk/2^32, where FmClk = 125MHz)
  Const TRUE = 1
  Const NULL = 0
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  CONST BZ1 = &H01                ' bit 0 mask for 8 bit word
  CONST BZ2 = &H00000001          ' bit 0 mask for 32 bit word
  
  SETPIN 22, DOUT         ' make pin 22 a digital output (used for AD9850 FQ_UD)
  PIN(22) = 0             ' and set it low to make sure FQ_UD is in idle state
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen is touched
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' Main program starts here
  InitAD9850    ' first go and initialise the AD9850
  ShowOScrn     ' then show the opening screen
  DO
    SELECT CASE OutFreq!  ' first get current freq as a string, with units
      CASE <1000
        CurrFreq$ = STR$(OutFreq!) + " Hz"  ' get current freq as a string
      CASE 1000 TO 999999.9
        CurrFreq$ = STR$(OutFreq!/1000) + " kHz"
      CASE >=1000000
        CurrFreq$ = STR$(OutFreq!/1000000) + " MHz"
    END SELECT
    FirstMenu   ' now display the first menu screen, get options
    IF Btn = 1 THEN ChangeFreq  ' button touched, so call ChangeFreq sub
  LOOP    ' End of main program loop  
END     ' End of program
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
SUB ShowOScrn ' sub to show the opening screen for five seconds
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 4, RGB(Cyan), DBlue
  TEXT 20, MM.VRes/4, "0", LM, 8, 1, Red, DBlue   ' this is the SC logo char/font
  TEXT MM.HRes/2, MM.VRes*3/4, "Simple AD9850", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*7/8, "Signal Generator", CM, 1, 2, White, DBlue
  PAUSE 5000  ' pause for five seconds before moving on
END SUB
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
Sub FirstMenu   ' sub to display the first (top level) menu screen
  IntFlag = 0   ' clear interrupt flag to begin
  CLS Black
  ' draw the button
  RBOX 0, MM.VRes*2/3, MM.HRes, MM.VRes/3, 10, Black, Bone
  COLOUR White, DBlue
  ' show current frequency, followed by button text
  TEXT 0, MM.VRes/16, "CURRENT FREQUENCY:", LM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes/3, CurrFreq$, CM, 1, 3, White, Black
  TEXT MM.HRes/2, MM.VRes*3/4, "Touch here to", CM, 1, 2, Black, Bone
  TEXT MM.HRes/2, MM.VRes*7/8, "Change Frequency", CM, 1, 2, Black, Bone
FM2: IF TOUCH(X) = -1 THEN GOTO FM2   ' keep looping until user touches screen
  
  IF TOUCH(Y) < MM.VRes/2 THEN GOTO FM2 ' loop back if button not touched
    Btn = 1         ' must have been first button
END SUB   ' then return, with Btn set to touched button number (1 or 2)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' subroutine to change the DDS output frequency
 SUB ChangeFreq
  IntFlag = 0   ' clear interrupt flag to begin 
  CLS Black
  TEXT 0,MM.VRes/16, "Current Frequency:", LM,1,1, White, Black
  TEXT MM.HRes/4, MM.VRes*3/16, CurrFreq$, CM, 1,2, White, Black
  TEXT 0, MM.VRes*5/16, "Enter New Freq:", LM, 1,1, White, Black
  BOX 0, MM.VRes*3/8, MM.HRes/2 -4, MM.VRes/8, 0, White, White
  TEXT MM.HRes/4, MM.VRes*7/16, SPACE$(9), CM, 1,2, Red, White
  TEXT 0, MM.VRes*5/8, "TOUCH OK BUTTON",LM,1,1, Green, Black
  TEXT 0, MM.VRes*11/16, "BELOW TO ENTER:", LM,1,1,Green, Black
  KPadDraw              ' now go draw keypad
  KPInStr$ = SPACE$(9)  ' clear the input string
  DO
    InCharFrmKP   ' then go get a character/string from the keypad (> KPStr$)
    SELECT CASE KPStr$
      CASE "OK"    ' OK button was touched
        IF KPInStr$ = SPACE$(9) THEN EXIT DO  ' if KPInStr$ null string, exit now
        NuFreq$ = LEFT$(KPInStr$, 9)  ' otherwise make KPInStr$ the new Frequency
        TEXT MM.HRes/4, MM.VRes*7/16, NuFreq$, CM, 1,2, Red, White  ' show it
        CurrFreq$ = NuFreq$       ' also update CurrFreq$ 
        OutFreq! = VAL(NuFreq$)   ' then update output frequency
        UpdateDDSFreq             ' & go change AD9850 frequency to match
        EXIT DO  ' then leave
      CASE "0" TO "9", "-", "."
        KPInStr$ = RIGHT$(KPInStr$, 8) ' a numeral, so shorten KPInStr$
        KPInStr$ = KPInStr$ + KPStr$  ' then add the new digit to it
        TEXT MM.HRes/4, MM.VRes*7/16, KPInStr$, CM, 1,2, Red, White ' & show it
      CASE CHR$(08)
        KPInStr$ = Left$(KPInStr$, LEN(KPInStr$) -1) 'if it's a backspace, blank last char
        TEXT MM.HRes/4, MM.VRes*7/16, KPInStr$ + " ", CM, 1,2, Red, White '& re-show
    END SELECT
    PAUSE 250
  LOOP
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to work out DDS Freq values from OutFreq! and then send to
' DDS as a 32-bit frequency word followed by an 8-bit control/phase word
SUB UpdateDDSFreq
    outData1 = &H00       ' the first 8 bit phase/control word
    FRegVal = OutFreq! / FoutFactor! ' get full FRegVal value
    outData2 = FRegVal AND &HFFFFFFFF  ' extract the lower 32 bits
    Send32bits            ' now send the main 32 bit word to the DDS
    Send8bits             ' followed by the 8 bit word
    ToggleFqud            ' finally toggle the FQ_UD line to load in
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to toggle AD-9850's FQ_UD line, to load serial data 
SUB ToggleFqud
  Pin(22) = 1   ' take FQ_UD line high
  PAUSE 0.001   ' pause for 1us (this may not be necessary)
  PIN(22) = 0   ' then bring FQ_UD line low again  
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to set IntFlag when touchscreen touched
SUB TchInt
  IntFlag = 1
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to reverse order of bits in outData1, -> revData1
SUB Rev8bits
  str1$ = BIN$(outData1,8)      ' get an 8-bit binary string from outData1
  revData1 = 0                  ' initialise revData1
  FOR Ctr = 8 TO 1 STEP -1      ' now work thru the binary digits
    IF MID$(str1$, Ctr, 1) = "1" THEN
      revData1 = revData1 OR &H01 ' make LSB a 1
    END IF
    revData1 = (revData1 << 1)  ' shift revData1 one bit to left 
  NEXT Ctr
  revData1 = (revData1 >> 1) AND &HFF   ' tidy up revData1
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to reverse order of bits in outData2, -> revData2
SUB Rev32bits
  str2$ = BIN$(outData2,32)     ' get a 32-bit binary string from OutData2
  revData2 = 0                  ' initialise revData2  
  FOR Ctr = 32 TO 1 STEP -1     ' now work thru the bin digits in str2$
    IF MID$(str2$, Ctr, 1) = "1" THEN
      revData2 = revData2 OR &H00000001 ' make LSB a 1
    END IF
    revData2 = (revData2 << 1)
  NEXT Ctr    ' reversed bits of outData2 -> revData2
  revData2 = (revData2 >> 1) AND &HFFFFFFFF    ' tidy up revData2
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to draw keypad on right, OK & backspace buttons lower left  
SUB KPadDraw  
  ' keypad boxes
  RBOX MM.HRes/2, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  ' boxes at lower left for backspace and OK buttons, plus legends
  RBOX 0, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4, 10, Black, Bone
  RBOX MM.HRes/6, MM.VRes*3/4, MM.HRes/3-4, MM.VRes/4, 10, Black, Bone
  TEXT MM.HRes/12, MM.VRes*7/8, "<-", CM, 1, 2, Black, Bone
  TEXT MM.HRes/3, MM.VRes*7/8, "OK", CM, 1, 3, Red, Bone
  ' legends for the keypad buttons
  TEXT MM.HRes*7/12, MM.VRes/8, "7", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes/8, "8", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes/8, "9", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*3/8, "4", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*3/8, "5", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*3/8, "6", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*5/8, "1", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*5/8, "2", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*5/8, "3", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*7/8, "-", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*7/8, "0", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*7/8, ".", CM, 1, 3, Black, Bone
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''  
SUB InCharFrmKP   ' sub to get a char from the touch keypad
                  ' (returns char in KPStr$)
  IntFlag = 0   ' clear interrupt flag to begin
ICFK2: IF TOUCH(X) = -1 THEN GOTO ICFK2 'wait until keypad touched
  ' but also loop back if no valid button was touched
  IF TOUCH(X) < MM.HRes/2 AND TOUCH(Y) < MM.VRes*3/4 THEN GOTO ICFK2
  SELECT CASE TOUCH(X)  ' now decide which button was touched
    CASE 0 TO MM.HRes/6-1
      KPStr$ = CHR$(08)  ' must have been backspace button
    CASE MM.HRes/6 TO MM.HRes/2-1
      KPStr$ = "OK"      ' must have been OK button
    CASE MM.HRes/2 TO MM.HRes*2/3-1  'first column of keypad itself
      SELECT CASE TOUCH(Y)  ' must be first column
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "7"   ' must be the 7 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "4"   ' must be the 4 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "1"   ' must be the 1 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "-"   ' must be the hyphen button
      END SELECT
    CASE MM.HRes*2/3 TO MM.HRes*5/6-1 'centre column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "8"   ' must be the 8 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "5"   ' must be the 5 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "2"   ' must be the 2 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "0"   ' must be the 0 button
      END SELECT
    CASE MM.HRes*5/6 TO MM.HRes   ' last column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "9"   ' must be the 9 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "6"   ' must be the 6 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "3"   ' must be the 3 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "."   ' must be the DP button
      END SELECT
  END SELECT
END SUB

'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'  subroutine to initialise the AD9833 DDS chip and then
'  start generating a 100,000Hz sinewave
SUB InitAD9850
      ToggleFqud            ' pulse FQ_UD line to enable serial mode
      outData2 = &H00000000 ' make outData2 32 zero bits
      Send32bits            ' then send freq word off to DDS
      outData1 = &H00       ' then make power down bit = 0
      Send8bits             ' send control/phase word off too
      ToggleFqud            ' then toggle FQ_UD to load into chip
      outData2 = &H346DC6   ' frequency word for Fout = 100kHz
      Send32bits            ' send it off
      Send8bits             ' then send outData1 again
      ToggleFqud            ' and toggle FQ_UD to load into chip
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to send an 8-bit word to the AD9850 via the SPI port
SUB Send8bits
  Rev8bits                ' first go reverse bit order (-> revData1)
  SPI OPEN 5000000,0, 8   ' the open SPI (speed, mode, bits)
  inData = SPI(revData1)  ' then send out the reversed 8 bit data word
  SPI CLOSE               ' finally close the SPI port again
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to send a 32-bit word to the AD9850 via the SPI port
SUB Send32bits
  Rev32bits               ' first go reverse bit order (-> revData2)
  SPI OPEN 5000000,0, 32  ' then open SPI (speed, mode, bits)
  inData = SPI(revData2)  ' then send out the reversed 32 bit freq word
  SPI CLOSE               ' finally close the SPI port again
END SUB

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' SCLOGOchar.bas
' Font type    : Numeric (1 characters)
' Font size    : 200x84 pixels
' Memory usage : 2100 bytes
' Font adapted from: http://www.rinkydinkelectronics.com/r_fonts.php

DefineFont #8
  013054C8 FF010000 F01F00F8 0F00E03F F8FF01F8 E0FF0300 FF00FE03 070000E0
  3F00FCFF 00C07FF0 FF07F81F FF0F00FC 00FE03F0 0000E0FF 00FEFF1F C07FF03F
  0FF81F00 1F00FEFF FE03F8FF 00E0FF00 FFFF7F00 7FF03F00 F81F00C0 00FFFF1F
  03FCFF3F E0FF01FF FFFF0100 F03F80FF 1F00C07F FFFF3FF8 FEFF7F80 FF01FF03
  FF0100E0 3FC0FFFF 00C07FE0 FF7FF01F FFFFC0FF 01FF07FF 0300E0FF E0FFC0FF
  80FFE07F 7FF03F00 FFC07FE0 FF07FF81 00C0FF01 FFC0FF03 FFE07FF0 F03F0080
  C1FFE07F 07FF83FF C0FF83FF C0FF0300 E07FE0FF 3F0080FF FFE07FF0 FF83FFC1
  FF83FF07 FF0300C0 7FE0FF80 0080FFE0 E07FF03F 83FFC1FF 83FF07FF 0300C0FF
  E0FF80FF 00FFC07F FFE03F00 FF81FFC0 FF0FFE03 0080FF83 FF81FF07 FFC1FFE0
  E07F0000 81FFC1FF 0FFE07FF 80FFC7FF 81FF0700 C1FFC0FF 7F0000FF FFC1FFE0
  FE07FF83 FFC7FF0F FF070080 FFC0FF81 0000FFC1 C1FFE07F 07FF83FF C7FF0FFE
  070080FF 000000FF 00FFC1FF FFE07F00 FF83FFC3 FF0FFE07 0080FFC7 0000FF07
  FE81FF00 C17F0000 03FF83FF 1FFC07FE 00FFEFFF 00FF0F00 83FF0100 FF0000FE
  0080FFC1 FC0FFE03 FFEFFF1F FF0F0000 FF010000 0000FE83 80FFC1FF 0FFE0700
  EFFF1FFC 0F0000FF 01E0FFFF 00FE83FF FFC1FF00 FE070080 FF1FFC0F 0000FFFF
  F0FFFF0F FE83FF01 C1FF0000 070080FF 1FFC0FFE 00FFFFFF FFFF0F00 03FF01FC
  FF0000FC 0000FF83 F80FFC07 FEFFFF3F FF070000 FF03FEFF 0100FC07 00FF83FF
  1FFC0700 FFFF3FF8 030000FE 03FFFFFF 00FC07FF FF83FF01 FC0F0000 FF3FF81F
  0000FEFF FFFFFF01 FC07FF83 83FF0100 0FFE03FF 3FF81FFC 00FEFFFF 07000000
  07FF83FF FF0100FC FE03FF83 F81FFC0F FEFFFF3F 00000000 FE83FF07 0100F807
  03FE07FF 1FF80FFE FFFF7FF0 000000FC 07FF0700 00F80FFE FE07FF03 F80FFC07
  FF7FF03F 0000FCFF FF070000 F80FFE07 07FF0300 1FFC07FE 7FF03FF8 00FCFFFB
  07FE1F00 0FFE07FF FF0300F8 FC07FE07 F03FF81F FCFFFB7F FE1F0000 FE07FF07
  0300F80F 07FE07FF 3FF81FFC FFFB7FF0 1F0000FC 07FF0FFE 00F00FFC FC0FFE03
  F01FFC07 F1FFE07F 0000F8FF FE0FFC3F F01FFC0F 0FFE0700 1FFC0FFC FFE07FF0
  00F8FFF1 0FFC3F00 1FFC0FFE FE0700F0 F80FFC0F E07FF03F F8FFF0FF FF3F0000
  FC0FFEFF 07FFFF1F 0FFC0FFE FFFF3FF8 FFF0FFE0 3F0000F8 0FFEFFFF FFFF1FFC
  FF0FFE07 FF3FF8FF E0FFE0FF 0000F8FF FCFFFF1F FF1FF80F 07FC07FF 1FF8FFFF
  FFC0FFFF 00F07FE0 FFFF0F00 3FF81FF8 FC0FFFFF F0FFFF03 81FFFF0F F07FE0FF
  FF070000 F81FE0FF 0FFFFF3F FFFF01FC FEFF07C0 7FE0FF01 030000F0 1F80FFFF
  FEFF3FF8 7F00FC0F FF0100FF C0FF01F8 0000F03F 00FFFF01 FF3FF81F 00FC0FFE
  0000FC3F FF01E07F 00E03FC0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 FEFF0100 3FF83F00 FF81FFE0 0000F8FF
  00000000 00000000 FF070000 F03F00FF 81FFE03F 00FCFFFF 00000000 00000000
  0F000000 7F80FFFF FFC17FF0 FEFFFF03 00000000 00000000 00000000 C0FFFF3F
  C17FF07F FFFF03FF 000000FF 00000000 00000000 FFFF7F00 7FF07FE0 FF03FFC1
  0080FFFF 00000000 00000000 FF7F0000 F07FF0FF 03FFC17F C0FFFFFF 00000000
  00000000 7F000000 7FE03FF0 FFC1FFE0 FF03FE03 000000C0 00000000 00000000
  E07FF0FF 83FFE0FF 03FE07FE 000080FF 00000000 00000000 7FF0FF00 FFE0FFE0
  FE07FE83 0080FF03 00000000 00000000 E0FF0000 E0FFE07F 07FE83FF 80FF03FE
  00000000 00000000 FF000000 FFE07FE0 FE83FFC1 FF03FE07 00000080 00000000
  00000000 C0FFE0FF 83FFC1FF 07FC07FE 000080FF 00000000 00000000 FFE0FF01
  FFC1FFC1 FC0FFC07 0000FF07 00000000 00000000 C0FF0100 C1FFC1FF 0FFC07FF
  00FF07FC 00000000 00000000 FF010000 FFC1FFC1 FC07FF81 FF07FC0F 00000000
  00000000 01000000 81FFC1FF 07FF83FF 07FC0FFC 000000FF 00000000 00000000
  00C0FF01 FFFFFF01 FF0FFC07 0000FFFF 00000000 00000000 80FF0300 FFFF0300
  1FF80FFE 00FEFFFF 00000000 00000000 FF030000 FF030080 F80FFEFF F8FFFF1F
  00000000 00000000 03000000 030080FF 0FFEFFFF FFFF1FF8 000000F0 00000000
  00000000 0080FF03 FEFFFF03 FF1FF80F 0000C0FF 00000000 00000000 80FF0300
  FFFF0300 1FF80FFE 0080FFFF 00000000 00000000 FF070000 FE07FF03 F01FFC0F
  0000F83F 00000000 00000000 07000000 07FF03FF 1FFC0FFE 00F83FF0 00000000
  00000000 00000000 FF03FF07 FC0FFE07 F83FF01F 00000000 00000000 00000000
  03FF0700 0FFC07FF 3FF01FFC 000000F8 00000000 00000000 FF070000 FC07FE07
  F01FFC1F 0000F83F 00000000 00000000 0F000000 0FFE07FE 3FF81FFC 00F07FE0
  00000000 00000000 00000000 FE07FE0F F81FFC0F F07FE03F 00000000 00000000
  00000000 07FE0F00 1FFC0FFE 7FE03FF8 000000F0 00000000 00000000 FE0F0000
  F80FFE07 E03FF83F 0000F07F 00000000 00000000 0F000000 0FFC0FFC 3FF83FF8
  00F07FE0 00000000 00000000 00000000 FC0FFC1F F03FF81F E0FFC07F 00000000
  00000000 00000000 0FFC1F00 3FF81FFC FFC07FF0 000000E0 00000000 00000000
  FF1F0000 F81FFCFF C07FF03F 0000E0FF 00000000 00000000 1F000000 1FF8FFFF
  7FF03FF0 00E0FFC0 00000000 00000000 00000000 F0FFFF0F F03FF01F E0FFC07F
  00000000 00000000 00000000 FFFF0700 7FF03FE0 FF81FFE0 000000C0 00000000
  00000000 FF010000 F03F80FF 81FFE07F 0000C0FF 00000000 00000000 00000000
  3F00FEFF FFE07FF0 00C0FF81 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
End DefineFont
